/***************************************************************
 *                   Materials Object Library                  *
 *Class cubic_elasticity_tensor : for cubic symmetry materials *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006,2008 COLLARD Christophe
 * copyright © 2006,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::cubic_elasticity_tensor
    \brief \f$ 4^{th} \f$ order elasticity tensor computation for materials with cubic symmetry

    \htmlonly 
    <FONT color="#838383">

    cubic elasticity tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    The stress tensor \f$ \boldsymbol\sigma \f$ is related to the strain tensor \f$ \boldsymbol{\EuScript{E}} \f$ through the \f$ 4^\text{th} \f$ order elasticity tensor \f$ \textbf{C} \f$ by : \n
    \f$ \sigma_{ij} = C_{ijkl} \EuScript{E}_{kl} \f$. \n
    By taking into account the symmetry of the strain and stress tensors, 
    \f$ \sigma_{ij} = C_{ijkl} \EuScript{E}_{kl} = C_{ijkl} \EuScript{E}_{lk} = \sigma_{ji} \f$, \n
    we get \f$C_{ijkl} = C_{jikl} = C_{ijlk} \f$.\n
    Moreover, if we take into account the symmetry property of the elasticity tensor \f$ C_{ijkl} = C_{klij} \f$, the \f$ 81 \f$ coefficients of the elasticity tensor reduces to 21 coefficients. \n
    Then, the relation between the stress tensor and the strain tensor can be written as : \n
    \f$ \begin{pmatrix} \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sigma_{12} \\ \sigma_{13} \\ \sigma_{23} \end{pmatrix} =
         \begin{pmatrix} c_{11} & c_{12} & c_{13} & c_{14} & c_{15} & c_{16} \\ 
                                & c_{22} & c_{23} & c_{24} & c_{25} & c_{26} \\
			        &        & c_{33} & c_{34} & c_{35} & c_{36} \\
			        & sym    &        & c_{44} & c_{45} & c_{46} \\
			        &        &        &        & c_{55} & c_{56} \\
			        &        &        &        &        & c_{66} \\
        \end{pmatrix} 
	\begin{pmatrix} \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ 2 \EuScript{E}_{12} \\ 2 \EuScript{E}_{13} \\ 2 \EuScript{E}_{23} \end{pmatrix} \f$ \n
    where \f$ \displaystyle c_{xy} = C_{XY} \f$, with the following relations between the tensor indices and the matrix indices  \n
    \f$ \text{\begin{tabular}{|c|c|c|c|c|c|c|c|c|c|}
    \hline 
    Tensor & 11 & 22 & 33 & 12 & 21 & 13 & 31 & 23 & 32 \\
    \hline
    Matrix & 1 & 2 & 3 & 4 & 4 & 5 & 5 & 6 & 6 \\
    \hline
    \end{tabular}} \f$


    For materials with cubic symmetry, the non zero elements of the elasticity tensor read \n
    \f$ \displaystyle C_{1111} = C_{2222} = C_{3333} = \frac{(1-\nu) E}{(1+\nu)(1-2\nu)} \f$ \n
    \f$ \displaystyle C_{1122} = C_{2211} = C_{1133} = C_{3311} = C_{2233} = C_{3322} = \frac{\nu E}{(1+\nu)(1-2\nu)} \f$ \n
    \f$ C_{1212} = C_{1221} = C_{2112} = C_{2121} = C_{1313} = C_{1331} = \mu \f$ \n
    \f$ C_{3113} = C_{3131} = C_{2323} = C_{2332} = C_{3223} = C_{3232} = \mu \f$ \n
    where : \n
    \f$ E \f$ : Young's modulus, \n
    \f$ \nu \f$ : Poisson ratio, \n
    \f$ \mu \f$ : Shear modulus. \n

    The associated reduced symmetric matrix reads : \n
    \f$ \begin{pmatrix} c_{11} & c_{12} & c_{13} & 0 & 0 & 0 \\ 
                        c_{21} & c_{22} & c_{23} & 0 & 0 & 0 \\
			c_{31} & c_{32} & c_{33} & 0 & 0 & 0 \\
			0 & 0 & 0 & c_{44} & 0 & 0 \\
			0 & 0 & 0 & 0 & c_{55} & 0 \\
			0 & 0 & 0 & 0 & 0 & c_{66}
        \end{pmatrix} \f$ \n
    with the following relations : \n
    \f$ \displaystyle c_{11} = c_{22} = c_{33} = \frac{(1-\nu) E}{(1+\nu)(1-2\nu)} \f$ \n
    \f$ \displaystyle c_{44} = c_{55} = c_{66} = \mu \f$ \n
    \f$ \displaystyle c_{12} = c_{21} = c_{13} = c_{31} = c_{23} = c_{32} = \frac{\nu E}{(1+\nu)(1-2\nu)} \f$. \n

    \author copyright \htmlonly &#169; \endhtmlonly 2006, 2008 Christophe COLLARD \n
            copyright \htmlonly &#169; 2006, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 2006-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type cubic elasticity tensor
#endif

#ifndef _cubic_elasticity_tensors_h
#define _cubic_elasticity_tensors_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__TENSORS4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__CUBIC_TENSORS_H)
#include "MateriOL++/cubic tensors.h"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//=======================================================================
template <class T> class cubic_elasticity_tensor : public cubic_tensor<T>
//=======================================================================
{
  using cubic_tensor<T>::C11;
  using cubic_tensor<T>::C44;
  using cubic_tensor<T>::C12;
  using cubic_tensor<T>::tensor_name;

  protected :
    T E, mu, nu;

  public :
    cubic_elasticity_tensor (string = "");  // default constructor
    cubic_elasticity_tensor (const tensor4<T>&, string = "");  // cast convertion
    ~cubic_elasticity_tensor () {}  // destructor

    string name(string = "");  // returns / computes tensor's name
    cubic_elasticity_tensor<T> inv ();
    virtual T Young (T);
    virtual T Shear (T);
    virtual T Poisson (T);

    //    returns constants
    T Young () const {return E;}
    T Shear () const {return mu;}
    T Poisson () const {return nu;}

    template <class Tf> friend ostream& operator << (ostream&, cubic_elasticity_tensor<Tf>&);
};


//=====Public methods for cubic elasticity tensor=======================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order cubic elasticity tensor.

  \param name tensor name
*/

//----------------------------------------------------------------------------------
template <class T> cubic_elasticity_tensor<T>::cubic_elasticity_tensor (string name)
//----------------------------------------------------------------------------------
{
  C11 = C44 = C12 = 0;
  E = nu = mu = 0;
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into a cubic elasticity tensor.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//-------------------------------------------------------------------------------------------------------
template <class T> cubic_elasticity_tensor<T>::cubic_elasticity_tensor (const tensor4<T>& t, string name)
//-------------------------------------------------------------------------------------------------------
{
  assert(t.dim1()==3 && t.dim2()==3 && t.dim3()==3 && t.dim4()==3);

  C11 = t(1,1,1,1);
  C12 = t(1,1,2,2);
  C44 = t(1,2,1,2);

  assert (C11 + C12 > 0);
  mu = C44;
  nu = C12 / (C11 + C12);
  E = (1 + nu) * (C11 - C12);
  tensor_name = name;
  cubic_tensor<T>::build_tensor();
  assert ((tensor4<T>)(*this) == t);
}


/*!
  \brief Changes and/or returns tensor name
  \param name tensor name
  \return tensor name
*/

//----------------------------------------------------------------------
template <class T> string cubic_elasticity_tensor<T>::name (string name)
//----------------------------------------------------------------------
{
  if (name != "") tensor_name = name;

  return tensor_name;
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order cubic tensor.

  Let \f$ (c_{11}, c_{44}, c_{12}) \f$ and \f$ (E, \mu, \nu) \f$ be the constants related to \f$ C \f$. \n
  If we note \f$ S = C^{-1} \f$, the non zero elements of \f$ S \f$ read : \n
  \f$ \displaystyle S_{1111} = S_{2222} = S_{3333} = \frac{1}{E} \f$ \n
  \f$ \displaystyle S_{1122} = S_{2211} = S_{1133} = S_{3311} = S_{2233} = S_{3322} = \frac{- \nu}{E} \f$ \n
  \f$ \displaystyle S_{1212} = S_{1221} = S_{2112} = S_{2121} = S_{1313} = S_{1331} = \frac{1}{\mu} \f$ \n
  \f$ \displaystyle S_{3113} = S_{3131} = S_{2323} = S_{2332} = S_{3223} = S_{3232} = \frac{1}{\mu} \f$ \n
  Moreover, if we note \f$ (E^\prime,\mu^\prime,\nu^\prime) \f$ the constants related to \f$ S \f$, we get : \n
  \f$ \displaystyle E^\prime = \frac{1}{c_{11}}, \mu^\prime = \frac{1}{\mu}, \nu^\prime = \frac{\nu}{\nu-1} \f$

  \return inverse tensor
*/

//------------------------------------------------------------------------------
template <class T> cubic_elasticity_tensor<T> cubic_elasticity_tensor<T>::inv ()
//------------------------------------------------------------------------------
{
  assert (E*nu*mu);

  T val = 1/E;
  //  cubic_tensor<T> Tinv(val,0.25/mu,-nu*val); // l'encodage (symtensor -> matrix) utilise n'est pas celui de la doc => laisser le 0.25 / mu !!
  cubic_elasticity_tensor<T> Tinv;
  Tinv.Poisson(nu/(nu-1));
  Tinv.Young(1/ C11);
  Tinv.Shear(0.25/mu);// l'encodage (symtensor -> matrix) utilise n'est pas celui de la doc => laisser le 0.25 / mu !!

  return Tinv;
}


/*!
  \brief Changes and returns \f$ E \f$ value (see detailed description) and rebuilds tensor if \f$ \mu \neq 0 \f$ and \f$ \nu \neq 0 \f$. \n
  \param scalar \f$ E \f$ value
  \return \f$ E \f$
*/

//---------------------------------------------------------------
template <class T> T cubic_elasticity_tensor<T>::Young (T scalar)
//---------------------------------------------------------------
{
  E = scalar;

  if (mu && nu)
    { T coef = 1/((1+nu) * (1 - 2*nu));
      C12 = nu * E * coef;
      C11 = (1-nu) * E * coef;
      C44 = mu;
      cubic_tensor<T>::build_tensor();
    }

  return E;
}


/*!
  \brief Changes and returns \f$ \mu \f$ value (see detailed description) and rebuilds tensor if \f$ E \neq 0 \f$ and \f$ \nu \neq 0 \f$. \n
  \param mu \f$ \mu \f$ value
  \return \f$ \mu \f$
*/

//---------------------------------------------------------------
template <class T> T cubic_elasticity_tensor<T>::Shear (T scalar)
//---------------------------------------------------------------
{
  mu = scalar;

  if (E && nu)
    { T coef = 1/((1+nu) * (1 - 2*nu));
      C12 = nu * E * coef;
      C11 = (1-nu) * E * coef;
      C44 = mu;
      cubic_tensor<T>::build_tensor();
    }

  return mu;
}


/*!
  \brief Changes and returns \f$ \nu \f$ value (see detailed description) and rebuilds tensor if \f$ E \neq 0 \f$ and \f$ \mu \neq 0 \f$. \n
  \param mu \f$ \nu \f$ value
  \return \f$ \nu \f$
*/

//-----------------------------------------------------------------
template <class T> T cubic_elasticity_tensor<T>::Poisson (T scalar)
//-----------------------------------------------------------------
{
  nu = scalar;

  if (E && mu)
    { T coef = 1/((1+nu) * (1 - 2*nu));
      C12 = nu * E * coef;
      C11 = (1-nu) * E * coef;
      C44 = mu;
      cubic_tensor<T>::build_tensor();
    }

  return nu;
}


//-------------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, cubic_elasticity_tensor<Tf>& tsr)
//-------------------------------------------------------------------------------------
{
  assert (tsr.E * tsr.nu * tsr.mu);

  s << tsr.tensor_name << endl;
  s << "Material constants : E = " << tsr.E << ", nu = " << tsr.nu << " mu = " << tsr.mu << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}


}


#endif
